/* Copyright (C) 2016-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef VNCCOMMONDECODERSUPPORTINGAPI_H_97923484074406319876074006167241293491
#define VNCCOMMONDECODERSUPPORTINGAPI_H_97923484074406319876074006167241293491

/**
 * \file vnccommondecodersupportingapi.h
 */

#include <vnccall.h>
#include <cdfdecoder/vnccommondecodertypes.h>

#ifdef __cplusplus
extern "C"
{
#endif


/** \defgroup supportingapis Supporting APIs */

/**
 * \brief Invoked by the decoder to write a message to the log. This is passed
 * synchronously to the SDK.
 * \ingroup supportingapis
 *
 * This supporting API may be invoked from any thread, but must not be invoked
 * after the decoder has returned from VNCCommonDecoderCallbackDestroyed.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param severity The severity of the log message. Typically, serious errors
 * are “0”, warnings are “10”, informational messages are “30”, and verbose
 * debugging is “100”.
 * \param tag The tag of the log message, as a NUL-terminated UTF-8 string.
 * Ownership stays with the decoder, and the pointer must be valid until this
 * call returns.
 * \param text The text of the log message, as a NUL-terminated UTF-8 string.
 * Ownership stays with the decoder, and the pointer must be valid until this
 * call returns.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPILog(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_int32_t severity,
    const char* tag,
    const char* text);

/**
 * \brief Invoked by the decoder to allocate a block of memory.
 * \ingroup supportingapis
 *
 * Whenever the decoder needs to allocate memory whose ownership is expected
 * to pass to the framework, the allocation must either be performed by
 * this API, or by VNCCommonDecoderSupportingAPIAllocString.
 *
 * If ownership of this block of memory is subsequently transferred to the
 * framework, then the framework will be responsible for freeing the memory.
 *
 * If ownership remains with the decoder (for example, if an error occurs
 * before the transfer can take place), then the decoder must free the memory
 * using VNCCommonDecoderFree.
 *
 * The start address of the returned memory is aligned to a word boundary, and
 * the memory is filled with zeroes.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \return void*. NULL if the allocation failed.
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param size The number of bytes to allocate.
 */
typedef void* VNCCALL
VNCCommonDecoderSupportingAPIAlloc(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_size_t size);

/**
 * \brief Invoked by the decoder to allocate a block of memory, and copy a
 * string into it.
 * \ingroup supportingapis
 *
 * Whenever the decoder needs to allocate memory whose ownership is expected
 * to pass to the framework, the allocation must either be performed by
 * this API, or by VNCCommonDecoderSupportingAPIAlloc.
 *
 * If ownership of this block of memory is subsequently transferred to the
 * framework, then the framework will be responsible for freeing the memory.
 *
 * If ownership remains with the decoder (for example, if an error occurs
 * before the transfer can take place), then the decoder must free the memory
 * using VNCCommonDecoderFreeString.
 *
 * The contents of the allocated memory will be equal to the supplied string,
 * including the NUL terminator.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \return char*. NULL if the allocation failed.
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param str The string to copy into the allocated memory.
 */
typedef char* VNCCALL
VNCCommonDecoderSupportingAPIAllocString(
    VNCCommonDecoderFrameworkContext frameworkContext,
    const char* str);

/**
 * \brief Called by the decoder to free memory which was allocated by
 * VNCCommonDecoderSupportingAPIAlloc, but whose ownership was not
 * transferred to the framework.
 * \ingroup supportingapis
 *
 * Payloads should NOT be provided to this method. To dispose of payloads, use
 * VNCCommonDecoderSupportingAPIStreamRecyclePayload.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param buffer The buffer to be freed.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIFree(
    VNCCommonDecoderFrameworkContext frameworkContext,
    void* buffer);

/**
 * \brief Called by the decoder to free memory which was allocated by
 * VNCCommonDecoderSupportingAPIAllocString, but whose ownership was not
 * transferred to the framework.
 * \ingroup supportingapis
 *
 * Payloads should NOT be provided to this method. To dispose of payloads, use
 * VNCCommonDecoderSupportingAPIStreamRecyclePayload.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param str The string to be freed.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIFreeString(
    VNCCommonDecoderFrameworkContext frameworkContext,
    char* str);

/**
 * \brief Called by the decoder to return a payload object to the framework.
 * \ingroup supportingapis
 *
 * Payload objects should be recycled as quickly as possible, to avoid excess
 * memory consumption.
 *
 * All payloads for a stream must have been returned by the time
 * VNCCommonDecoderSupportingAPIStreamDestroyCompleted is invoked.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 * \param payload The payload to be recycled.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamRecyclePayload(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId,
    VNCCommonDecoderStreamPayload* payload);

/**
 * \brief Called by the decoder to request that the source of the current
 * stream limit the video framerate.
 * \ingroup supportingapis
 *
 * This may be called at any time while the stream exists, whether or not it is
 * playing.
 *
 * The requestStreamFrameRateLimitSupported field in the
 * VNCCommonDecoderSDKAttributes struct should be checked before invoking this
 * API.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 * \param framerate The desired framerate, in frames per second.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamRequestFrameRateLimit(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId,
    vnc_uint32_t framerate);

/**
 * \brief Called by the decoder to request that the source of the current
 * video stream sends an IDR frame.
 * \ingroup supportingapis
 *
 * This may be called at any time while the stream exists, whether or not it is
 * playing.
 *
 * The requestStreamIDRFrameSupported field in the
 * VNCCommonDecoderSDKAttributes struct should be checked before invoking this
 * API.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamRequestIDRFrame(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId);

/**
 * \brief Called by the decoder to request that the source of the current
 * stream resets the stream.
 * \ingroup supportingapis
 *
 * This may be called at any time while the stream exists, whether or not it is
 * playing.
 *
 * The requestStreamReset*Supported fields in the
 * VNCCommonDecoderSDKAttributes struct should be checked before invoking this
 * API.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \return vnc_bool_t. Set to vnc_false if this stream is known not to support
 * stream reset requests. Set to vnc_true otherwise.
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamRequestReset(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId);

/**
 * \brief Called by the decoder to check whether a particular set of features
 * are licensed.
 * \ingroup supportingapis
 *
 * This method allows you to ensure that your custom decoders are usable only
 * in certain applications.  The feature check succeeds if the SDK is
 * authorized to use at least one of the requested features.  The features are
 * searched for in the licenses available to the SDK.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \return VNCCommonDecoderError. Set to VNCCommonDecoderErrorNone if the check
 * was successful. If not, the result should be ignored.
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param featureIdList Pointer to the first element of the list of feature IDs
 * which should be checked.
 * \param featureIdCount The number of feature IDs in the list.
 * \param out_resultList Pointer to the first element of a list where the
 * results should be stored. This will be modified by the framework.
 */
typedef VNCCommonDecoderError VNCCALL
VNCCommonDecoderSupportingAPILocalFeatureCheck(
    VNCCommonDecoderFrameworkContext frameworkContext,
    const unsigned int* featureIdList,
    vnc_size_t featureIdCount,
    int* out_resultList);

/**
 * \brief Called by the decoder to notify the SDK that the decoder cannot
 * continue processing a particular stream. For example, the stream data may be
 * unrecoverably malformed, or there may be a temporary failure in the relevant
 * decoder hardware.
 * \ingroup supportingapis
 *
 * The SDK should take steps to end the stream, and perhaps the whole
 * connection if necessary. The stream should not simply be restarted without
 * user interaction.
 *
 * Decoder callbacks relating to the stream will continue to be forwarded for
 * an undefined period of time. The decoder may ignore the stream after
 * invoking this API, but must return all payload objects at the appropriate
 * time.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 * \param error The error which occurred.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamError(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId,
    VNCCommonDecoderError error);

/**
 * \brief Called by the decoder to notify the SDK that the decoder cannot
 * continue processing any stream. For instance, the decoder may have
 * encountered an unexpected internal failure from which it cannot recover, or
 * there may have been a permanent failure in the decoding hardware.
 * \ingroup supportingapis
 *
 * The SDK should take steps to end all streams, and perhaps the whole
 * connection if necessary. The stream/decoder should not simply be restarted
 * without user interaction.
 *
 * Decoder callbacks relating to all streams will continue to be forwarded for
 * an undefined period of time. The decoder may ignore all streams after
 * invoking this API, but must return all payload objects at the appropriate
 * times.
 *
 * This supporting API must be invoked from the SDK thread.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param error The error which occurred.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIError(
    VNCCommonDecoderFrameworkContext frameworkContext,
    VNCCommonDecoderError error);

/**
 * \brief Called by the decoder to provide some data to the SDK user.
 * \ingroup supportingapis
 *
 * This operates on the same principle as VNCCommonDecoderCallbackPropertySet,
 * but in the reverse direction, allowing the decoder to provide arbitrary
 * callbacks to the SDK user.
 *
 * This supporting API must be invoked from the SDK thread, and will block
 * until the SDK user has returned from the corresponding callback.
 *
 * \return VNCCommonDecoderError. This will be passed unchanged from the
 * SDK user to the decoder.
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param propertyKey The identifier of the property being set. This will be
 * passed unchanged from the decoder to the SDK user.
 * \param propertyType The data type pointed to by propertyValue. This will be
 * passed unchanged from the decoder to the SDK user.
 * \param propertyValue The new value of the property being set. This will be
 * passed unchanged from the decoder to the SDK user.
 */
typedef VNCCommonDecoderError VNCCALL
VNCCommonDecoderSupportingAPISetFrameworkUserProperty(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_int32_t propertyKey,
    VNCCommonDecoderDataType propertyType,
    void* propertyValue);

/**
 * \brief Called by the decoder to notify the SDK of the resolution
 * of a particular video stream.
 * \ingroup supportingapis
 *
 * This API should be invoked when the first frame of video is first decoded,
 * and from then on should be invoked whenever the resolution of the stream
 * changes.
 *
 * For performance reasons, this API should not be called unnecessarily, and
 * should only be called during actual changes.
 *
 * If the framework or SDK detect that the stream does not exist, or does not
 * contain any video, this callback should be ignored, but a log message may be
 * written.
 *
 * This supporting API must be invoked from the SDK thread, and may block
 * until the SDK has returned from the corresponding callback.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 * \param width The video stream width, in pixels.
 * \param height The video stream height, in pixels.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamResolutionUpdated(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId,
    vnc_uint16_t width,
    vnc_uint16_t height);

/**
 * \brief Called by the decoder in response to
 * VNCCommonDecoderCallbackStreamCreated, to inform the SDK that the decoder
 * now considers the stream to be in the “Stopped” state.
 * \ingroup supportingapis
 *
 * Failure to invoke this API in response to
 * VNCCommonDecoderCallbackStreamCreated could cause the framework or SDK to
 * hang forever. It must be called even in cases where a failure occurred.
 *
 * State change supporting API calls must occur in the same order as the
 * original state change decoder callbacks.
 *
 * This supporting API must be invoked from the SDK thread. This can occur from
 * any decoder callback, including the one which originally triggered the state
 * change.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamCreateCompleted(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId);

/**
 * \brief Called by the decoder in response to
 * VNCCommonDecoderCallbackStreamStarted, to inform the SDK that the decoder
 * now considers the stream to have moved from the “Stopped” state to the
 * “Playing” state.
 * \ingroup supportingapis
 *
 * Failure to invoke this API in response to
 * VNCCommonDecoderCallbackStreamStarted could cause the framework or SDK to
 * hang forever. It must be called even in cases where a failure occurred.
 *
 * State change supporting API calls must occur in the same order as the
 * original state change decoder callbacks.
 *
 * This supporting API must be invoked from the SDK thread. This can occur from
 * any decoder callback, including the one which originally triggered the state
 * change.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamStartCompleted(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId);

/**
 * \brief Called by the decoder in response to
 * VNCCommonDecoderCallbackStreamStopped, to inform the SDK that the decoder
 * now considers the stream to have moved from the “Playing” state to the
 * “Stopped” state.
 * \ingroup supportingapis
 *
 * Assuming that the stream has not encountered an error, the decoder must have
 * finished rendering all payloads before invoking this supporting API. Note
 * that this only applies to payloads received before the triggering
 * VNCCommonDecoderCallbackStreamStopped callback. Also, it does not apply if
 * the stopImmediately parameter was set during the
 * VNCCommonDecoderCallbackStreamStopped callback.
 *
 * Failure to invoke this API in response to
 * VNCCommonDecoderCallbackStreamStopped could cause the framework or SDK to
 * hang forever. It must be called even in cases where a failure occurred.
 *
 * State change supporting API calls must occur in the same order as the
 * original state change decoder callbacks.
 *
 * This supporting API must be invoked from the SDK thread. This can occur from
 * any decoder callback, including the one which originally triggered the state
 * change.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamStopCompleted(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId);

/**
 * \brief Called by the decoder in response to
 * VNCCommonDecoderCallbackStreamDestroyed, to inform the SDK that the decoder
 * now considers the stream to have moved from the “Stopped” state to the
 * “Destroyed” state.
 * \ingroup supportingapis
 *
 * Regardless of any stream errors, the decoder must have recycled all payloads
 * for this stream before invoking this supporting API.
 *
 * Failure to invoke this API in response to
 * VNCCommonDecoderCallbackStreamDestroyed could cause the framework or SDK to
 * hang forever. It must be called even in cases where a failure occurred.
 *
 * State change supporting API calls must occur in the same order as the
 * original state change decoder callbacks.
 *
 * This supporting API must be invoked from the SDK thread. This can occur from
 * any decoder callback, including the one which originally triggered the state
 * change.
 *
 * \param frameworkContext The context pointer which was given to the entry
 * point.
 * \param streamId The unique ID of the relevant stream.
 */
typedef void VNCCALL
VNCCommonDecoderSupportingAPIStreamDestroyCompleted(
    VNCCommonDecoderFrameworkContext frameworkContext,
    vnc_uint64_t streamId);


/**
 * \brief APIs provided by the SDK for calling by the decoder.
 * \ingroup supportingapis
 */
typedef struct {

  /**
   * Invoked by the decoder to write a message to the log.
   */
  VNCCommonDecoderSupportingAPILog* vncCommonDecoderSupportingAPILog;

  /**
   * Invoked by the decoder to allocate a block of memory.
   */
  VNCCommonDecoderSupportingAPIAlloc* vncCommonDecoderSupportingAPIAlloc;

  /**
   * Invoked by the decoder to copy a string.
   */
  VNCCommonDecoderSupportingAPIAllocString* vncCommonDecoderSupportingAPIAllocString;

  /**
   * Called by the decoder to free memory.
   */
  VNCCommonDecoderSupportingAPIFree* vncCommonDecoderSupportingAPIFree;

  /**
   * Called by the decoder to free memory.
   */
  VNCCommonDecoderSupportingAPIFreeString* vncCommonDecoderSupportingAPIFreeString;

  /**
   * Called by the decoder to return a payload object to the framework.
   */
  VNCCommonDecoderSupportingAPIStreamRecyclePayload* vncCommonDecoderSupportingAPIStreamRecyclePayload;

  /**
   * Called by the decoder to request that the source of the current
   * stream limit the video framerate.
   */
  VNCCommonDecoderSupportingAPIStreamRequestFrameRateLimit* vncCommonDecoderSupportingAPIStreamRequestFrameRateLimit;

  /**
   * Called by the decoder to request that the source of the current
   * stream sends an IDR frame.
   */
  VNCCommonDecoderSupportingAPIStreamRequestIDRFrame* vncCommonDecoderSupportingAPIStreamRequestIDRFrame;

  /**
   * Called by the decoder to request that the source of the current
   * stream resets the stream.
   */
  VNCCommonDecoderSupportingAPIStreamRequestReset* vncCommonDecoderSupportingAPIStreamRequestReset;

  /**
   * Called by the decoder to check whether a particular set of features
   * are licensed.
   */
  VNCCommonDecoderSupportingAPILocalFeatureCheck* vncCommonDecoderSupportingAPILocalFeatureCheck;

  /**
   * Called by the decoder to notify the SDK that the decoder cannot
   * continue processing a particular stream.
   */
  VNCCommonDecoderSupportingAPIStreamError* vncCommonDecoderSupportingAPIStreamError;

  /**
   * Called by the decoder to notify the SDK that the decoder cannot
   * continue processing any stream.
   */
  VNCCommonDecoderSupportingAPIError* vncCommonDecoderSupportingAPIError;

  /**
   * Called by the decoder to provide some data to the SDK user.
   */
  VNCCommonDecoderSupportingAPISetFrameworkUserProperty* vncCommonDecoderSupportingAPISetFrameworkUserProperty;

  /**
   * Called by the decoder to notify the SDK of the resolution
   * of a particular video stream.
   */
  VNCCommonDecoderSupportingAPIStreamResolutionUpdated* vncCommonDecoderSupportingAPIStreamResolutionUpdated;

  /**
   * Called by the decoder in response to VNCCommonDecoderCallbackStreamCreated.
   */
  VNCCommonDecoderSupportingAPIStreamCreateCompleted* vncCommonDecoderSupportingAPIStreamCreateCompleted;

  /**
   * Called by the decoder in response to VNCCommonDecoderCallbackStreamStarted.
   */
  VNCCommonDecoderSupportingAPIStreamStartCompleted* vncCommonDecoderSupportingAPIStreamStartCompleted;

  /**
   * Called by the decoder in response to VNCCommonDecoderCallbackStreamStopped.
   */
  VNCCommonDecoderSupportingAPIStreamStopCompleted* vncCommonDecoderSupportingAPIStreamStopCompleted;

  /**
   * Called by the decoder in response to
   * vncCommonDecoderCallbackStreamDestroyed
   */
  VNCCommonDecoderSupportingAPIStreamDestroyCompleted* vncCommonDecoderSupportingAPIStreamDestroyCompleted;

} VNCCommonDecoderSupportingAPI;


#ifdef __cplusplus
}
#endif

#endif // VNCCOMMONDECODERSUPPORTINGAPI_H_97923484074406319876074006167241293491

